-- ============================================================================
-- AutoOpen System - Advanced Automatic Door & Gate Control for FS25
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - Intelligent player/vehicle detection
-- - Configurable user attributes per trigger
-- - Farm-based security checks
-- - Counter-based collision system
-- - Multiplayer synchronization
-- - Automatic timer system
-- - Complete GUI settings integration
-- - StopBlocker MP-Synchronisation
-- ============================================================================

local modDirectory = g_currentModDirectory
local modName = g_currentModName

-- Load debug system first
source(Utils.getFilename("src/core/AutoOpenDebug.lua", modDirectory))

debugPrint("AutoOpen System v1.1.0.0 - Loading...")

-- Load core modules in correct order
local sourceFiles = {
    -- Events (load early for network functionality)
    "src/events/AutoOpenEvent.lua",
    "src/events/AutoOpenStopBlockerEvent.lua",  -- *** NEUE DATEI FÜR MP-FIX ***
    
    -- GUI Manager System (before Core for settings)
    "src/gui/AutoOpenSettingsManager.lua",
    "src/gui/AutoOpenManager.lua",
    
    -- Core Logic (uses Manager for settings)
    "src/core/AutoOpenCore.lua",
	
    -- stopBlocker funktion laden (MP-VERSION)
    "src/core/AutoOpenStopBlocker.lua",  
	
    -- Hooks (load last, orchestrates everything)
    "src/core/AutoOpenHooks.lua",	
}

-- Load all modules sequentially with error handling
for _, sourceFile in ipairs(sourceFiles) do
    local fullPath = Utils.getFilename(sourceFile, modDirectory)
    
    if fileExists(fullPath) then
        local success, errorMsg = pcall(function()
            source(fullPath)
        end)
        
        if success then
            debugPrint("Loaded: " .. sourceFile)
        else
            debugPrint("ERROR loading " .. sourceFile .. ": " .. tostring(errorMsg))
        end
    else
        debugPrint("FILE NOT FOUND: " .. sourceFile)
    end
end

-- ============================================================================
-- System Initialization
-- ============================================================================

---Check if AutoOpen system is loaded
local function isLoaded()
    return g_currentMission ~= nil and g_currentMission.autoOpen ~= nil
end

---Initialize the AutoOpen system
local function load(mission)
    local success, errorMsg = pcall(function()
        debugPrint("AutoOpen System - Initializing...")
        
        -- Initialize the AutoOpen system
        AutoOpenHooks:initialize(mission, g_i18n, modName, modDirectory)
        
        debugPrint("AutoOpen System - Ready (v1.1.0.0 MP-Support aktiv)")
    end)
    
    if not success then
        debugPrint("CRITICAL ERROR loading AutoOpen System: " .. tostring(errorMsg))
    end
end

---Cleanup AutoOpen system on shutdown
local function unload()
    local success, errorMsg = pcall(function()
        if not isLoaded() then
            return
        end

        debugPrint("AutoOpen System - Shutting down...")
        
        AutoOpenHooks:cleanup()
        
        debugPrint("AutoOpen System - Cleanup complete")
    end)
    
    if not success then
        debugPrint("ERROR during AutoOpen System cleanup: " .. tostring(errorMsg))
    end
end

-- ============================================================================
-- Hook Installation
-- ============================================================================

---Install system hooks
local function init()
    debugPrint("AutoOpen System - Installing hooks...")
    
    -- Mission hooks
    FSBaseMission.delete = Utils.appendedFunction(FSBaseMission.delete, unload)
    Mission00.load = Utils.prependedFunction(Mission00.load, load)
    
    debugPrint("AutoOpen System - Hooks installed")
end

-- Global identifier for other mods
g_autoOpenModName = modName

-- Initialize system
init()

-- print("AutoOpen System v1.1.0.0 by GulliDeckel - Loaded successfully with Multiplayer Support")